<?php
/**
 * Admin settings page template
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Get PHP configuration values
$max_upload_size = ini_get('upload_max_filesize');
$max_post_size = ini_get('post_max_size');
$max_file_uploads = ini_get('max_file_uploads');

// Convert to bytes for comparison
function convert_to_bytes($value) {
    $value = trim($value);
    $last = strtolower($value[strlen($value)-1]);
    $value = (int)$value;
    switch($last) {
        case 'g': $value *= 1024;
        case 'm': $value *= 1024;
        case 'k': $value *= 1024;
    }
    return $value;
}

$max_upload_bytes = convert_to_bytes($max_upload_size);
$max_post_bytes = convert_to_bytes($max_post_size);
$effective_max_mb = min($max_upload_bytes, $max_post_bytes) / 1024 / 1024;

// Get current settings
$settings = get_option('spai_contact_form_settings', array(
    'delete_data_on_uninstall' => false,
    'enable_file_uploads' => true,
    'max_file_size' => 10,
    'max_attachments' => 2,
    'allowed_file_types' => array('pdf', 'doc', 'docx', 'jpg', 'jpeg', 'png'),
    'enable_honeypot' => true,
    'enable_recaptcha' => false,
    'recaptcha_site_key' => '',
    'recaptcha_secret_key' => ''
));

// Handle form submission
if (isset($_POST['submit'])) {
    check_admin_referer('spai_settings_nonce');
    
    $settings = array(
        'delete_data_on_uninstall' => isset($_POST['delete_data_on_uninstall']),
        'enable_file_uploads' => isset($_POST['enable_file_uploads']),
        'max_file_size' => intval($_POST['max_file_size']),
        'max_attachments' => intval($_POST['max_attachments']),
        'allowed_file_types' => array_map('trim', explode(',', $_POST['allowed_file_types'])),
        'enable_honeypot' => isset($_POST['enable_honeypot']),
        'enable_recaptcha' => isset($_POST['enable_recaptcha']),
        'recaptcha_site_key' => sanitize_text_field($_POST['recaptcha_site_key']),
        'recaptcha_secret_key' => sanitize_text_field($_POST['recaptcha_secret_key'])
    );
    
    // Validate against PHP limits
    $warnings = array();
    
    if ($settings['max_file_size'] > $effective_max_mb) {
        $settings['max_file_size'] = floor($effective_max_mb);
        $warnings[] = sprintf(__('Max file size was adjusted to %d MB to match PHP limits.', 'spai-contact-form'), $settings['max_file_size']);
    }
    
    if ($settings['max_attachments'] > $max_file_uploads) {
        $settings['max_attachments'] = $max_file_uploads;
        $warnings[] = sprintf(__('Max attachments was adjusted to %d to match PHP max_file_uploads.', 'spai-contact-form'), $settings['max_attachments']);
    }
    
    update_option('spai_contact_form_settings', $settings);
    
    echo '<div class="notice notice-success"><p>' . __('Settings saved.', 'spai-contact-form') . '</p></div>';
    
    if (!empty($warnings)) {
        echo '<div class="notice notice-warning">';
        foreach ($warnings as $warning) {
            echo '<p>' . $warning . '</p>';
        }
        echo '</div>';
    }
}
?>

<div class="wrap">
    <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
    
    <div class="notice notice-info">
        <p><strong><?php _e('Server Upload Limits:', 'spai-contact-form'); ?></strong></p>
        <ul style="list-style: disc; margin-left: 20px;">
            <li><?php printf(__('PHP upload_max_filesize: %s', 'spai-contact-form'), '<code>' . $max_upload_size . '</code>'); ?></li>
            <li><?php printf(__('PHP post_max_size: %s', 'spai-contact-form'), '<code>' . $max_post_size . '</code>'); ?></li>
            <li><?php printf(__('PHP max_file_uploads: %s files', 'spai-contact-form'), '<code>' . $max_file_uploads . '</code>'); ?></li>
            <li><?php printf(__('Effective maximum file size: %s MB', 'spai-contact-form'), '<strong>' . number_format($effective_max_mb, 0) . '</strong>'); ?></li>
        </ul>
        <p><em><?php _e('Your settings cannot exceed these PHP limits. Contact your hosting provider to increase them if needed.', 'spai-contact-form'); ?></em></p>
    </div>
    
    <form method="post" action="">
        <?php wp_nonce_field('spai_settings_nonce'); ?>
        
        <table class="form-table">            <tr>
                <th scope="row"><?php _e('Delete Data on Uninstall', 'spai-contact-form'); ?></th>
                <td>
                    <label>
                        <input type="checkbox" name="delete_data_on_uninstall" value="1" <?php checked($settings['delete_data_on_uninstall']); ?>>
                        <?php _e('Remove all plugin data when uninstalling', 'spai-contact-form'); ?>
                    </label>
                </td>
            </tr>
            
            <tr>
                <th scope="row"><?php _e('File Uploads', 'spai-contact-form'); ?></th>
                <td>
                    <label>
                        <input type="checkbox" name="enable_file_uploads" value="1" <?php checked($settings['enable_file_uploads']); ?>>
                        <?php _e('Enable file uploads', 'spai-contact-form'); ?>
                    </label>
                </td>
            </tr>
            
            <tr>
                <th scope="row"><?php _e('Max File Size (MB)', 'spai-contact-form'); ?></th>
                <td>
                    <input type="number" name="max_file_size" value="<?php echo esc_attr($settings['max_file_size']); ?>" min="1" max="100">
                    <p class="description">
                        <?php 
                        printf(
                            __('PHP limits: upload_max_filesize = %s, post_max_size = %s (Effective max: %s MB)', 'spai-contact-form'),
                            $max_upload_size,
                            $max_post_size,
                            number_format($effective_max_mb, 0)
                        );
                        ?>
                    </p>
                </td>
            </tr>
            
            <tr>
                <th scope="row"><?php _e('Max Attachments', 'spai-contact-form'); ?></th>
                <td>
                    <input type="number" name="max_attachments" value="<?php echo esc_attr($settings['max_attachments']); ?>" min="1" max="10">
                    <p class="description">
                        <?php 
                        printf(
                            __('Maximum number of files per submission. PHP max_file_uploads = %s', 'spai-contact-form'),
                            $max_file_uploads
                        );
                        ?>
                    </p>
                </td>
            </tr>
            
            <tr>
                <th scope="row"><?php _e('Allowed File Types', 'spai-contact-form'); ?></th>
                <td>
                    <input type="text" name="allowed_file_types" value="<?php echo esc_attr(implode(', ', $settings['allowed_file_types'])); ?>" class="regular-text">
                    <p class="description"><?php _e('Comma-separated list of file extensions', 'spai-contact-form'); ?></p>
                </td>
            </tr>
            
            <tr>
                <th scope="row"><?php _e('Spam Protection', 'spai-contact-form'); ?></th>
                <td>
                    <label>
                        <input type="checkbox" name="enable_honeypot" value="1" <?php checked($settings['enable_honeypot']); ?>>
                        <?php _e('Enable honeypot spam protection', 'spai-contact-form'); ?>
                    </label>
                </td>
            </tr>
            
            <tr>
                <th scope="row"><?php _e('reCAPTCHA v3', 'spai-contact-form'); ?></th>
                <td>
                    <label>
                        <input type="checkbox" name="enable_recaptcha" value="1" <?php checked($settings['enable_recaptcha']); ?>>
                        <?php _e('Enable Google reCAPTCHA v3', 'spai-contact-form'); ?>
                    </label>
                </td>
            </tr>
            
            <tr>
                <th scope="row"><?php _e('reCAPTCHA v3 Site Key', 'spai-contact-form'); ?></th>
                <td>
                    <input type="text" name="recaptcha_site_key" value="<?php echo esc_attr($settings['recaptcha_site_key']); ?>" class="regular-text">
                </td>
            </tr>
            
            <tr>
                <th scope="row"><?php _e('reCAPTCHA v3 Secret Key', 'spai-contact-form'); ?></th>
                <td>
                    <input type="text" name="recaptcha_secret_key" value="<?php echo esc_attr($settings['recaptcha_secret_key']); ?>" class="regular-text">
                </td>
            </tr>
        </table>
        
        <?php submit_button(); ?>
    </form>
</div>