<?php
/**
 * Base class for all integrations
 *
 * @package SPAI_Contact_Form
 */

abstract class SPAI_Integration_Base {
    
    /**
     * Integration name
     */
    abstract public function get_name();
    
    /**
     * Integration description
     */
    abstract public function get_description();
    
    /**
     * Get settings fields for this integration
     */
    abstract public function get_settings_fields();
    
    /**
     * Get field mapping configuration
     */
    abstract public function get_fields_config();
    
    /**
     * Process submission
     */
    abstract public function process_submission($data, $mapping);
    
    /**
     * Test connection
     */
    abstract public function test_connection();
    
    /**
     * Transform field value based on target field type
     */
    protected function transform_value($value, $target_type, $options = array()) {
        switch ($target_type) {
            case 'text':
                return is_array($value) ? implode(', ', $value) : strval($value);
                
            case 'number':
                return floatval($value);
                
            case 'date':
                if (!empty($value)) {
                    $format = isset($options['format']) ? $options['format'] : 'Y-m-d';
                    $date = strtotime($value);
                    return $date ? date($format, $date) : $value;
                }
                return $value;
                
            case 'boolean':
                return filter_var($value, FILTER_VALIDATE_BOOLEAN);
                
            case 'json':
                return json_encode($value);
                
            default:
                return $value;
        }
    }
    
    /**
     * Map form fields to integration fields
     */
    protected function map_fields($form_data, $mapping) {
        $mapped_data = array();
        
        foreach ($mapping as $form_field => $integration_config) {
            if (isset($form_data[$form_field])) {
                $value = $form_data[$form_field];
                
                if (isset($integration_config['transform'])) {
                    $value = $this->transform_value(
                        $value,
                        $integration_config['transform'],
                        isset($integration_config['transform_options']) ? $integration_config['transform_options'] : array()
                    );
                }
                
                $mapped_data[$integration_config['field']] = $value;
            }
        }
        
        return $mapped_data;
    }
    
    /**
     * Log integration activity
     */
    protected function log($message, $type = 'info', $data = array()) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            
            if (!empty($data)) {
            }
        }
    }
}
