<?php
/**
 * Field types and configuration manager
 *
 * @package SPAI_Contact_Form
 */

class SPAI_Field_Manager {
    
    /**
     * Get available field types
     */
    public static function get_field_types() {
        return array(
            'text' => array(
                'label' => __('Text', 'spai-contact-form'),
                'icon' => 'dashicons-editor-textcolor',
                'settings' => array('placeholder', 'default_value', 'min_length', 'max_length', 'pattern')
            ),
            'textarea' => array(
                'label' => __('Textarea', 'spai-contact-form'),
                'icon' => 'dashicons-editor-paragraph',
                'settings' => array('placeholder', 'default_value', 'rows', 'min_length', 'max_length')
            ),
            'email' => array(
                'label' => __('Email', 'spai-contact-form'),
                'icon' => 'dashicons-email',
                'settings' => array('placeholder', 'default_value')
            ),
            'phone' => array(
                'label' => __('Phone', 'spai-contact-form'),
                'icon' => 'dashicons-phone',
                'settings' => array('placeholder', 'default_value', 'country_code')
            ),
            'number' => array(
                'label' => __('Number', 'spai-contact-form'),
                'icon' => 'dashicons-calculator',
                'settings' => array('placeholder', 'default_value', 'min', 'max', 'step')
            ),
            'select' => array(
                'label' => __('Dropdown', 'spai-contact-form'),
                'icon' => 'dashicons-arrow-down-alt2',
                'settings' => array('placeholder', 'options', 'default_value', 'multiple')
            ),
            'radio' => array(
                'label' => __('Radio Buttons', 'spai-contact-form'),
                'icon' => 'dashicons-marker',
                'settings' => array('options', 'default_value', 'inline')
            ),
            'checkbox' => array(
                'label' => __('Checkbox', 'spai-contact-form'),
                'icon' => 'dashicons-yes',
                'settings' => array('options', 'default_values', 'inline')
            ),
            'multichoice' => array(
                'label' => __('Multi Choice', 'spai-contact-form'),
                'icon' => 'dashicons-tag',
                'settings' => array('options', 'default_values', 'placeholder')
            ),
            'consent_checkbox' => array(
                'label' => __('Consent Checkbox', 'spai-contact-form'),
                'icon' => 'dashicons-yes-alt',
                'settings' => array('label_text', 'default_checked', 'required_text')
            ),
            'date' => array(
                'label' => __('Date', 'spai-contact-form'),
                'icon' => 'dashicons-calendar-alt',
                'settings' => array('placeholder', 'default_value', 'min_date', 'max_date', 'date_format')
            ),
            'file' => array(
                'label' => __('File Upload', 'spai-contact-form'),
                'icon' => 'dashicons-media-default',
                'settings' => array('allowed_types', 'max_size', 'multiple', 'max_files')
            ),
            'hidden' => array(
                'label' => __('Hidden', 'spai-contact-form'),
                'icon' => 'dashicons-hidden',
                'settings' => array('default_value')
            ),
            'section' => array(
                'label' => __('Section', 'spai-contact-form'),
                'icon' => 'dashicons-editor-insertmore',
                'settings' => array('heading', 'description')
            )
        );
    }
    
    /**
     * Get default field configuration
     */
    public static function get_default_field($type) {
        $defaults = array(
            'id' => uniqid('field_'),
            'type' => $type,
            'label' => '',
            'name' => '',
            'required' => false,
            'css_class' => '',
            'help_text' => '',
            'conditional_logic' => array(
                'enabled' => false,
                'rules' => array()
            )
        );
        
        // Add type-specific defaults
        switch ($type) {
            case 'textarea':
                $defaults['rows'] = 5;
                break;
            case 'select':
            case 'radio':
            case 'checkbox':
            case 'multichoice':
                $defaults['options'] = array();
                break;
            case 'consent_checkbox':
                $defaults['label_text'] = 'By checking this box, you agree to receive text messages. Standard message and data rates may apply. Reply STOP to opt out at any time.';
                $defaults['default_checked'] = true;
                $defaults['required_text'] = 'You must agree to receive text messages.';
                break;
            case 'file':
                $defaults['allowed_types'] = array('pdf', 'doc', 'docx', 'jpg', 'jpeg', 'png');
                $defaults['max_size'] = 10; // MB
                $defaults['multiple'] = false;
                break;
            case 'date':
                $defaults['date_format'] = 'Y-m-d';
                break;
        }
        
        return $defaults;
    }
    
    /**
     * Validate field configuration
     */
    public static function validate_field($field) {
        $errors = array();
        
        // Required fields
        if (empty($field['label'])) {
            $errors[] = __('Field label is required.', 'spai-contact-form');
        }
        
        if (empty($field['name'])) {
            $errors[] = __('Field name is required.', 'spai-contact-form');
        }
        
        // Name validation
        if (!empty($field['name']) && !preg_match('/^[a-zA-Z][a-zA-Z0-9_]*$/', $field['name'])) {
            $errors[] = __('Field name must start with a letter and contain only letters, numbers, and underscores.', 'spai-contact-form');
        }
        
        // Type-specific validation
        if (in_array($field['type'], array('select', 'radio', 'checkbox', 'multichoice')) && empty($field['options'])) {
            $errors[] = __('Options are required for this field type.', 'spai-contact-form');
        }
        
        return $errors;
    }
    
    /**
     * Sanitize field value based on type
     */
    public static function sanitize_field_value($value, $field) {
        switch ($field['type']) {
            case 'email':
                return sanitize_email($value);
            
            case 'textarea':
                return sanitize_textarea_field($value);
            
            case 'number':
                return floatval($value);
            
            case 'phone':
                return preg_replace('/[^0-9+\-\s\(\)]/', '', $value);
            
            case 'date':
                return sanitize_text_field($value);
            
            case 'checkbox':
            case 'multichoice':
                if (is_array($value)) {
                    return array_map('sanitize_text_field', $value);
                }
                return $value ? true : false;
            
            case 'consent_checkbox':
                return $value ? true : false;
            
            case 'file':
                // File handling is done separately
                return $value;
            
            case 'select':
                if (is_array($value)) {
                    return array_map('sanitize_text_field', $value);
                }
                return sanitize_text_field($value);
            
            default:
                return sanitize_text_field($value);
        }
    }
}
