<?php
/**
 * Fired during plugin activation
 *
 * @package SPAI_Contact_Form
 */

class SPAI_Activator {
    
    /**
     * Actions to perform on plugin activation
     */
    public static function activate() {
        // Create database tables
        self::create_tables();
        
        // Create upload directory
        self::create_upload_directory();
        
        // Set default options
        self::set_default_options();
        
        // Clear permalinks
        flush_rewrite_rules();
    }
    
    /**
     * Create plugin database tables
     */
    private static function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Forms table
        $table_forms = $wpdb->prefix . 'spai_forms';
        $sql_forms = "CREATE TABLE $table_forms (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            form_name varchar(255) NOT NULL,
            form_fields longtext NOT NULL,
            form_settings longtext NOT NULL,
            active tinyint(1) DEFAULT 1,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) $charset_collate;";
        
        // Submissions table
        $table_submissions = $wpdb->prefix . 'spai_submissions';
        $sql_submissions = "CREATE TABLE $table_submissions (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            form_id mediumint(9) NOT NULL,
            submission_data longtext NOT NULL,
            integration_status longtext NOT NULL,
            file_data longtext,
            ip_address varchar(45),
            user_agent text,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY form_id (form_id)
        ) $charset_collate;";
        
        // Integration queue table
        $table_queue = $wpdb->prefix . 'spai_integration_queue';
        $sql_queue = "CREATE TABLE $table_queue (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            submission_id mediumint(9) NOT NULL,
            integration_type varchar(50) NOT NULL,
            payload longtext NOT NULL,
            status enum('pending','processing','completed','failed') DEFAULT 'pending',
            attempts int DEFAULT 0,
            last_error text,
            external_id varchar(255),
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            processed_at datetime,
            PRIMARY KEY (id),
            KEY submission_id (submission_id),
            KEY status (status)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql_forms);
        dbDelta($sql_submissions);
        dbDelta($sql_queue);
        
        // Store current version
        update_option('spai_contact_form_version', SPAI_CONTACT_FORM_VERSION);
    }
    
    /**
     * Create upload directory for form files
     */
    private static function create_upload_directory() {
        $upload_dir = wp_upload_dir();
        $spai_upload_dir = $upload_dir['basedir'] . '/' . SPAI_CONTACT_FORM_UPLOAD_DIR;
        
        if (!file_exists($spai_upload_dir)) {
            wp_mkdir_p($spai_upload_dir);
            
            // Add .htaccess to protect directory
            $htaccess_content = "Options -Indexes\n";
            $htaccess_content .= "<FilesMatch \"\\.(php|php\\d|phtml)$\">\n";
            $htaccess_content .= "Order Allow,Deny\n";
            $htaccess_content .= "Deny from all\n";
            $htaccess_content .= "</FilesMatch>\n";
            
            file_put_contents($spai_upload_dir . '/.htaccess', $htaccess_content);
        }
    }
    
    /**
     * Set default plugin options
     */
    private static function set_default_options() {
        // General settings
        add_option('spai_contact_form_settings', array(
            'delete_data_on_uninstall' => false,
            'enable_file_uploads' => true,
            'max_file_size' => 10, // MB
            'allowed_file_types' => array('pdf', 'doc', 'docx', 'jpg', 'jpeg', 'png'),
            'enable_honeypot' => true,
            'enable_recaptcha' => false,
            'recaptcha_site_key' => '',
            'recaptcha_secret_key' => ''
        ));
    }
}
